
import React, { useState } from 'react';
import { UserRole, User } from '../types';
import ClipboardListIcon from './icons/ClipboardListIcon';
import ChartBarIcon from './icons/ChartBarIcon';
import DocumentDuplicateIcon from './icons/DocumentDuplicateIcon';
import CalendarIcon from './icons/CalendarIcon';
import UserIcon from './icons/UserIcon';

interface HeaderProps {
  currentUserRole: UserRole;
  setCurrentUserRole: (role: UserRole) => void;
  currentView: 'list' | 'dashboard' | 'grds' | 'calendar' | 'users';
  setCurrentView: (view: 'list' | 'dashboard' | 'grds' | 'calendar' | 'users') => void;
  users: User[];
  currentUser: User | null;
  onSelectUser: (userId: string) => void;
  onOpenProfile: () => void;
  onLogout: () => void;
}

const Header: React.FC<HeaderProps> = ({ currentUserRole, setCurrentUserRole, currentView, setCurrentView, users, currentUser, onSelectUser, onOpenProfile, onLogout }) => {
  const roles = Object.values(UserRole);
  const [menuOpen, setMenuOpen] = useState(false);

  return (
    <header className="bg-primary-dark/80 backdrop-blur-sm sticky top-0 z-20 shadow-lg shadow-primary-dark/50">
      <div className="container mx-auto px-4 sm:px-6 lg:px-8 py-4 flex flex-col sm:flex-row justify-between items-center">
        <h1 className="text-2xl sm:text-3xl font-bold text-white mb-4 sm:mb-0">
          <span className="text-primary-light">MY</span> - Gestão de Entrega
        </h1>
        <div className="flex items-center gap-4">
            <div className="flex items-center space-x-1 bg-primary-medium/20 p-1 rounded-full">
                <button
                    onClick={() => setCurrentView('list')}
                    className={`p-2 rounded-full transition-all duration-300 ${
                        currentView === 'list' ? 'bg-primary-light text-primary-dark shadow-md' : 'text-gray-300 hover:bg-primary-medium/50 hover:text-white'
                    }`}
                    aria-label="Visualizar Lista"
                >
                    <ClipboardListIcon className="w-5 h-5" />
                </button>
                 <button
                    onClick={() => setCurrentView('calendar')}
                    className={`p-2 rounded-full transition-all duration-300 ${
                        currentView === 'calendar' ? 'bg-primary-light text-primary-dark shadow-md' : 'text-gray-300 hover:bg-primary-medium/50 hover:text-white'
                    }`}
                    aria-label="Visualizar Calendário"
                >
                    <CalendarIcon className="w-5 h-5" />
                </button>
                <button
                    onClick={() => setCurrentView('dashboard')}
                    className={`p-2 rounded-full transition-all duration-300 ${
                        currentView === 'dashboard' ? 'bg-primary-light text-primary-dark shadow-md' : 'text-gray-300 hover:bg-primary-medium/50 hover:text-white'
                    }`}
                    aria-label="Visualizar Dashboard"
                >
                    <ChartBarIcon className="w-5 h-5" />
                </button>
                 <button
                    onClick={() => setCurrentView('grds')}
                    className={`p-2 rounded-full transition-all duration-300 ${
                        currentView === 'grds' ? 'bg-primary-light text-primary-dark shadow-md' : 'text-gray-300 hover:bg-primary-medium/50 hover:text-white'
                    }`}
                    aria-label="Visualizar GRDs"
                >
                    <DocumentDuplicateIcon className="w-5 h-5" />
                </button>
            </div>
            {currentUserRole === UserRole.MASTER && (
              <button
                onClick={() => setCurrentView('users')}
                className={`px-3 py-2 rounded-full transition-all duration-300 ${
                  currentView === 'users' ? 'bg-primary-light text-primary-dark shadow-md' : 'text-gray-300 hover:bg-primary-medium/50 hover:text-white'
                }`}
                aria-label="Gerenciar Usuários"
              >
                Usuários
              </button>
            )}
            <div className="relative flex items-center gap-3">
              <div className="text-sm text-gray-300">
                <span className="font-semibold">{currentUser?.nome || 'Usuário'}</span>
                <span className="mx-1">•</span>
                <span>{currentUser?.role || ''}</span>
              </div>
              <button
                onClick={() => setMenuOpen(v => !v)}
                className="p-1 rounded-full hover:bg-primary-medium/50 transition-all"
                aria-label="Abrir menu de perfil"
                title="Perfil"
              >
                {currentUser?.photo_url ? (
                  <img src={currentUser.photo_url} alt="Avatar" className="w-10 h-10 rounded-full object-cover border border-primary-medium/60" />
                ) : (
                  <UserIcon className="w-10 h-10 text-gray-300" />
                )}
              </button>
              {menuOpen && (
                <div className="absolute right-0 top-full mt-2 bg-primary-dark border border-primary-medium/50 rounded-xl shadow-lg w-48 z-50">
                  <button
                    className="w-full text-left px-4 py-2 text-white hover:bg-primary-medium/40"
                    onClick={() => { setMenuOpen(false); onOpenProfile(); }}
                  >
                    Editar perfil
                  </button>
                  <button
                    className="w-full text-left px-4 py-2 text-red-300 hover:bg-red-500/30"
                    onClick={() => { setMenuOpen(false); onLogout(); }}
                  >
                    Sair da Aplicação
                  </button>
                </div>
              )}
            </div>
        </div>
      </div>
    </header>
  );
};

export default Header;
